/**
 * \file: AlsaAudioCommon.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: CarPlay
 *
 * \author: C. Gellner / ADIT/SW1 / cgellner@de.adit-jv.com
 *
 * \copyright (c) 2013-2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <memory.h>
#include <adit_logging.h>
#include "AlsaAudioCommon.h"

using namespace std;

LOG_IMPORT_CONTEXT(cply);

#define NUM_TRY_RECOVER 10
#define SILENCE_PREFILL_DEFAULT 2
#define BUFFER_PERIODS_DEFAULT 2
#define getErrAsStr(X) ((X == EINVAL) ? "EINVAL" : (X == EPERM) ? "EPERM" : "UNKNOWNERR")

namespace adit { namespace carplay
{

AlsaAudioCommon::AlsaAudioCommon(bool inDumpConfiguration, bool inVerboseLogging, const IConfiguration& Config)
{
    dump_config = inDumpConfiguration;
    verbose = inVerboseLogging;

    pcm = NULL;
    silence_buffer = NULL;

    /* initial settings */
    dir = SND_PCM_STREAM_PLAYBACK;
    format = SND_PCM_FORMAT_S16_LE;
    channels = 2;
    rate = 48000;
    ideal_period_ms = 50;
    buffer_ideal_periods = -1;
    prefill_ms = -1;
    bytes_per_frame = 0;

    init_tout = ideal_period_ms + 20;
    recover_delay_ms = 0;
    startup_xfer = 0;

    setup = false;
    started = false;
    broken = false;
    wait_done = false;

    aborted = 0;

    if(Config.GetNumber("disable-real-time-priority-audio", 0))
    {
        setThreadPrio = false;
    }
    else
    {
        setThreadPrio = true;
        threadPrio = Config.GetNumber("audio-threads-real-time-priority", 61);
    }
}

AlsaAudioCommon::~AlsaAudioCommon()
{
    ReleasePCM();
}

#define DumpHwSwParams(func,stream,attoutput) \
        if (attoutput != NULL) {\
            snd_output_printf(attoutput, "Dumping HwParams for %s at %s", stream, func);\
            snd_pcm_hw_params_dump(hw_params, attoutput);\
            snd_output_printf(attoutput, "Dumping SwParams for %s at %s", stream, func);\
            snd_pcm_sw_params_dump(sw_params, attoutput);\
            snd_output_flush (attoutput);\
        }

#define LogSndStdErrorAndReturn(err_,func,stream_,attoutput) \
    if (err_ < 0) {\
        LOG_ERROR((cply, "%s failed with err %d (%s) for stream %s",\
            func, err_, snd_strerror(err_), stream_));\
        DumpHwSwParams(func, stream_, attoutput);\
        if (attoutput != NULL)\
            snd_output_close(attoutput);\
        return err_;\
    }

void AlsaAudioCommon::SetDir(snd_pcm_stream_t dir_)
{
    dir = dir_;
}
void AlsaAudioCommon::SetFormat(snd_pcm_format_t format_)
{
    format = format_;
}
void AlsaAudioCommon::SetChannels(unsigned int channels_)
{
    channels = channels_;
}
void AlsaAudioCommon::SetRate(unsigned int rate_)
{
    rate = rate_;
}
void AlsaAudioCommon::SetInitTout(int tout_)
{
    init_tout = tout_;
}
void AlsaAudioCommon::SetIdealPeriodMs(unsigned int ideal_period_ms_)
{
    ideal_period_ms = ideal_period_ms_;
}
void AlsaAudioCommon::SetBufferIdealPeriods (int buffer_ideal_periods_)
{
    if (buffer_ideal_periods_ >= 0) {
        if (buffer_ideal_periods_ < 2) {
            LOG_ERROR((cply, "Buffer ideal periods has to be >= 2\n"));
            return;
        }
        buffer_ideal_periods = buffer_ideal_periods_;
    } else {
        buffer_ideal_periods = -1;
    }
}
void AlsaAudioCommon::SetPrefillMs(int prefill_ms_)
{
    if (prefill_ms_ >= 0) {
        prefill_ms = prefill_ms_;
    } else {
        prefill_ms = -1;
    }
}
void AlsaAudioCommon::SetRecoverDelayMs(unsigned int recover_delay_ms_)
{
    recover_delay_ms = recover_delay_ms_;
}

snd_pcm_uframes_t AlsaAudioCommon::GetIdealPeriodMs()
{
    if (setup == false)
        return 0;
    return ideal_period_size;
}

snd_pcm_uframes_t AlsaAudioCommon::GetPrefillMs()
{
    if (setup == false)
        return 0;
    return silence_prefill;
}

int AlsaAudioCommon::SetupPCM(const char *pcmname)
{
    int err;
    snd_pcm_hw_params_t *hw_params;
    snd_pcm_sw_params_t *sw_params;
    unsigned int i;
    unsigned int chmin, chmax;
    int prefill_ms_local;
    int buffer_periods_local;
    const unsigned char dividers[] =
    { 2, 3, 5, 6, 7, 10, 0 }; /*zero terminated list of dividers*/
    unsigned int div;
    snd_output_t *output = NULL;

    /*setup acces types in preferred order*/
    const snd_pcm_access_t pcm_access[] =
    { SND_PCM_ACCESS_MMAP_INTERLEAVED, SND_PCM_ACCESS_RW_INTERLEAVED };

    setup = false;

    if (pcm != NULL)
    {
        LOG_ERROR((cply, "Failed to setup %s as pcm is already present", pcmname));
        return -EINVAL;
    }

    if (dump_config == true)
    {
        err = snd_output_stdio_attach(&output, stdout, 0);
        if (err < 0)
        {
            LOG_WARN((cply, "Attaching stdout to snd_output failed"));
            err = 0;
            output = NULL;
        }
    }

    snd_pcm_hw_params_alloca(&hw_params);
    snd_pcm_sw_params_alloca(&sw_params);
    LOG_INFO((cply, "AlsaAudioCommon CFG %s rate %u, channels %u, dir %s, format %s, " \
            "ideal period ms %u, buffer ideal periods %d, prefill ms %d", pcmname, rate, channels, snd_pcm_stream_name (dir),
            snd_pcm_format_name (format), ideal_period_ms, buffer_ideal_periods, prefill_ms));

    err = snd_pcm_open(&pcm, pcmname, dir, SND_PCM_NONBLOCK);
    LogSndStdErrorAndReturn(err, "snd_pcm_open", pcmname, output);

    err = snd_pcm_hw_params_any(pcm, hw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_any", pcmname, output);

    err = snd_pcm_hw_params_set_format(pcm, hw_params, format);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_format", pcmname, output);

    chmin = channels;
    chmax = channels;
    err = snd_pcm_hw_params_set_channels_minmax(pcm, hw_params, &chmin, &chmax);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_channels_min", pcmname, output);
    if ((chmin != channels) || (chmax != channels))
    {
        LOG_ERROR((cply, "Failed to set min max channels to %u (got %u to %u)", channels, chmin,
                chmax));
        return -EINVAL;
    }

    err = snd_pcm_hw_params_set_rate(pcm, hw_params, rate, 0);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_rate", pcmname, output);

    for (i = 0; i < sizeof(pcm_access); i++)
    {
        err = snd_pcm_hw_params_set_access(pcm, hw_params, pcm_access[i]);
        if (err == 0)
        {
            access = pcm_access[i];
            break;
        }
    }
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_access", pcmname, output);

    ideal_period_size = (ideal_period_ms * rate) / 1000;

    if (ideal_period_size == 0) {
        LOG_ERROR((cply, "Invalid ideal period size\n"));
        return -EINVAL;
    }

    silence_prefill = 0;
    if (dir == SND_PCM_STREAM_PLAYBACK) {
        prefill_ms_local = prefill_ms;

        /* using default */
        if (prefill_ms_local < 0) {
            LOG_INFO((cply, "Using default %d periods for silence prefill for %s.\n", SILENCE_PREFILL_DEFAULT, pcmname));
            prefill_ms_local = SILENCE_PREFILL_DEFAULT * ideal_period_ms;
        }

        /* if silence is a multiple of ideal period ms use a differnt calc to avoid rounding issues */
        if ((prefill_ms_local % ideal_period_ms) == 0) {
            silence_prefill = ideal_period_size * (prefill_ms_local / ideal_period_ms);
        } else {
            silence_prefill = (prefill_ms_local * rate) / 1000;
        }
    } else {
        if (prefill_ms > 0)
            LOG_WARN((cply, "Silence prefill is only valid for playback - setting will be ignored for %s\n", pcmname));
    }
    LOG_INFO((cply, "AlsaAudioCommon CFG %s ideal period size %lu, silence prefill frames %lu\n",
                    pcmname, ideal_period_size, silence_prefill));

    err = -EINVAL;
    for (i = 0; (dividers[i] != 0) && err; i++)
    {
        div = dividers[i];
        period_size = ideal_period_size;
        do
        {
            snd_pcm_uframes_t psize_min = period_size;
            snd_pcm_uframes_t psize_max = period_size;
            int mindir = -1;
            int maxdir = +1;
            err = snd_pcm_hw_params_set_period_size_minmax(pcm, hw_params, &psize_min, &mindir,
                    &psize_max, &maxdir);
            if(err) {
                period_size = ideal_period_size / div;
                div *= dividers[i];
            }
        } while ((period_size >= 16) && err);/*reasonable limit*/
    }
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_period_size", pcmname, output);

    buffer_periods_local = buffer_ideal_periods;
    /* using default */
    if (buffer_periods_local < 0) {
        LOG_INFO((cply, "Using default %d periods for buffer for %s.\n", BUFFER_PERIODS_DEFAULT, pcmname));
        buffer_periods_local = BUFFER_PERIODS_DEFAULT;
    }

    if (buffer_periods_local < 2) {
        LOG_ERROR((cply, "Invalid buffer configuration for %s. Buffer has to be at least two periods\n", pcmname));
        return -EINVAL;
    }

    buffer_size = buffer_periods_local * ideal_period_size;
    /*extra space for larger silence prefill*/
    if (silence_prefill > (buffer_size - ideal_period_size)) {
        unsigned int silence_periods = ((silence_prefill - 1) / ideal_period_size) + 1;
        buffer_periods_local = silence_periods + 1;
        buffer_size = buffer_periods_local * ideal_period_size;
        LOG_INFO((cply, "Extending buffer for %s to %d periods.\n", pcmname, buffer_periods_local));
    }

    err = snd_pcm_hw_params_set_buffer_size_min(pcm, hw_params, &buffer_size);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_buffer_size_min", pcmname, output);

    err = snd_pcm_hw_params_set_buffer_size_near(pcm, hw_params, &buffer_size);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_buffer_size_near", pcmname, output);

    LOG_INFO((cply, "AlsaAudioCommon CFG %s period size %lu, buffer size %lu", pcmname, period_size,
            buffer_size));

    err = snd_pcm_hw_params(pcm, hw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params", pcmname, output);

    err = snd_pcm_sw_params_current(pcm, sw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params_current", pcmname, output);

    if (dir == SND_PCM_STREAM_CAPTURE)
        err = snd_pcm_sw_params_set_start_threshold(pcm, sw_params, 1);
    else
        err = snd_pcm_sw_params_set_start_threshold(pcm, sw_params,
                ideal_period_size + silence_prefill);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params_set_start_threshold", pcmname, output);

    err = snd_pcm_sw_params_set_avail_min(pcm, sw_params, ideal_period_size);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params_set_avail_min", pcmname, output);

    err = snd_pcm_sw_params(pcm, sw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params", pcmname, output);

    bytes_per_frame = snd_pcm_format_size(format, channels);

    LOGD_DEBUG((cply, "AlsaAudioCommon CFG %s complete", pcmname));

    DumpHwSwParams("configuration complete", pcmname, output);

    if (output != NULL)
    {
        snd_output_printf(output, "Dumping pcm configuration for %s", pcmname);\
        snd_pcm_dump(pcm, output);
        snd_output_flush(output);
        snd_output_close(output);
    }

    ReleaseBuffers();

    err = CreateBuffers();
    if (err < 0)
        return err;

    started = false;
    setup = true;
    broken = false;
    wait_done = false;
    startup_xfer = 0;
    aborted = 0;

    return 0;
}

void AlsaAudioCommon::ReleasePCM()
{
    if (pcm != NULL)
    {
        LOG_INFO((cply, "ReleasePCM %s", snd_pcm_name(pcm)));

        if (started == true)
            snd_pcm_drop(pcm);

        snd_pcm_close(pcm);
    }
    pcm = NULL;

    setup = false;

    ReleaseBuffers();
}

int AlsaAudioCommon::CreateBuffers()
{
    unsigned int silence_buffer_size = ideal_period_size * channels;

    if (dir != SND_PCM_STREAM_PLAYBACK)
        return 0;

    silence_buffer = malloc(snd_pcm_format_size(format, silence_buffer_size));

    if (silence_buffer == NULL)
    {
        LOG_ERROR((cply, "AlsaAudioCommon failed to create buffers"));

        ReleaseBuffers();

        return -ENOMEM;
    }

    snd_pcm_format_set_silence(format, silence_buffer, silence_buffer_size);

    return 0;
}

void AlsaAudioCommon::ReleaseBuffers()
{
    if (silence_buffer != NULL)
        free(silence_buffer);

    silence_buffer = NULL;
}

int AlsaAudioCommon::PrefillSilence()
{
    int err = 0;
    snd_pcm_uframes_t silence_remain;
    silence_remain = silence_prefill;
    if (verbose) {
        LOGD_VERBOSE((cply, "Prefill silence %s", snd_pcm_name(pcm)));
    }
    while ((silence_remain > 0) && (err == 0))
    {
        snd_pcm_uframes_t chunk = silence_remain;
        if (chunk > ideal_period_size)
            chunk = ideal_period_size;

        err = RwBuffer(silence_buffer, chunk);

        silence_remain -= chunk;
        if (err < 0)
        {
            LOG_ERROR((cply, "RW SILENCE ERROR %d (%s) on dev %s", err, snd_strerror(err),
                    snd_pcm_name(pcm)));
            break;
        }
    }
    return err;
}

#define MAX_BLOCK_MS 100 /*maximum time to block*/

int AlsaAudioCommon::WaitPCM()
{
    int err;
    int wait_time_ms = (wait_done==true)?(ideal_period_ms + 20):init_tout;
    int wait_tries = wait_time_ms/MAX_BLOCK_MS;

    if (wait_time_ms % MAX_BLOCK_MS)
    {
        wait_tries++;
    }
    wait_time_ms /= wait_tries;

    while (wait_tries--)
    {
        err = snd_pcm_wait(pcm, wait_time_ms);
        if (aborted)
        {
            LOGD_DEBUG((cply, "WAIT aborted on dev %s", snd_pcm_name(pcm)));
            return (err == -EINTR)?err:-EINVAL;
        }
        if (err <= 0)
        {
            if (err < 0)
            {
                LOG_ERROR((cply, "WAIT ERR %d (%s) on dev %s", err, snd_strerror(err),
                        snd_pcm_name(pcm)));
                return err;
            }
            if (!wait_tries)
            {
                LOG_ERROR((cply, "WAIT TOUT dev %s -> end", snd_pcm_name(pcm)));
                if (wait_done)
                {
                    /* timeout during streaming. try to recover */
                    return -ENODATA;
                }
                else
                {
                    /* initial timeout occur -> abort (no recover) */
                    return -EINVAL;
                }
            }
            LOGD_DEBUG((cply, "WAIT TOUT dev %s -> retry", snd_pcm_name(pcm)));
        }
        else
            break;
    }

    return 0;
}

int AlsaAudioCommon::RwBuffer(void *buffer, snd_pcm_uframes_t xfer_size_)
{
    unsigned int retry_done = 0;
    int err = 0;
    char *buf_ptr;
    snd_pcm_uframes_t xfer_size;

    if (verbose)
    {
        LOGD_VERBOSE((cply, "RW buffer %s - buffer %p, size %ld", snd_pcm_name(pcm),
                        buffer, xfer_size_));
    }

    buf_ptr = (char*) buffer;
    xfer_size = xfer_size_;

    while ((xfer_size > 0) && (err == 0))
    {
        do
        {/*EINTR should not occur, as we operate in nonblocking mode*/
            switch (access)
            {
            case SND_PCM_ACCESS_MMAP_INTERLEAVED:
                if (dir == SND_PCM_STREAM_PLAYBACK)
                {
                    err = snd_pcm_mmap_writei(pcm, buf_ptr, xfer_size);
                }
                else
                {
                    err = snd_pcm_mmap_readi(pcm, buf_ptr, xfer_size);
                }
                break;
            case SND_PCM_ACCESS_RW_INTERLEAVED:
                if (dir == SND_PCM_STREAM_PLAYBACK)
                {
                    err = snd_pcm_writei(pcm, buf_ptr, xfer_size);
                }
                else
                {
                    err = snd_pcm_readi(pcm, buf_ptr, xfer_size);
                }
                break;

            default:
                LOG_ERROR((cply, "rw: invalid access %d", access));
                err = -EINVAL;
                break;
            }
        } while (err == -EINTR);

        if (err >= 0)
        {
            if (err > (int) xfer_size)
            {
                LOG_ERROR((cply, "invalid size %d", err));
                err = -EINVAL;
            }
            else
            {
		if ((err > 0) && (wait_done == false)) {
                        startup_xfer += err;
			if (dir == SND_PCM_STREAM_CAPTURE){
				wait_done = true;
			} else {
				if (startup_xfer > buffer_size)
					wait_done = true;
			}
		}
                buf_ptr += err * bytes_per_frame;
                xfer_size -= err;
                err = 0;
            }
        }

        if (err == -EAGAIN)
        {
            /*
             Non period aligned write is buggy in libasound < 1.0.27:
             Writing smaller buffers than period size can result in 100% CPU load, as snd_pcm_wait()
             succeeds, while following write access returs -EAGAIN.
             Fixed with commit 'PCM: Avoid busy loop in snd_pcm_write_areas() with rate plugin'.
             As a workaround we do a short sleep if we get repeated EAGAIN's
             */
            if (retry_done != 0)
                usleep(1000);
            retry_done = 1;
            err = WaitPCM();
        }
    }

    return err;

}

int AlsaAudioCommon::Recover(int reset)
{
    int err;

    LOGD_DEBUG((cply, "Recovering %s", snd_pcm_name(pcm)));

    started = false;

    if (reset)
    {
        wait_done = false;
        startup_xfer = 0;
    }
    err = snd_pcm_drop(pcm);
    if (err < 0)
    {
        return err;
    }
    err = snd_pcm_prepare(pcm);

    return err;
}

int AlsaAudioCommon::ReadWrite(void *buffer, size_t len)
{
    int err = 0;
    int tmp_err;
    int retry_cnt = 0;
    snd_pcm_uframes_t buffer_frames;

    if (setup == false)
    {
        LOG_ERROR((cply, "ReadWrite must not be called before Setup is completed"));
        return -EINVAL;
    }
    if (broken != false)
    {
        LOG_ERROR((cply, "Stream is broken. In order to retry close and call Setup again"));
        return -EINVAL;
    }

    if ((len % bytes_per_frame) != 0)
    {
        LOG_ERROR((cply, "Buffer size %zd is not aligned to the bytes per frame %d given by " \
                "format %s and channels %u", len, bytes_per_frame, snd_pcm_format_name(format),
                channels));
        return -EINVAL;
    }

    buffer_frames = len / bytes_per_frame;

    if (verbose)
    {
        LOGD_VERBOSE((cply, "New buffer for %s : Buffer addr 0x%p, frames %ld",
                        snd_pcm_name(pcm), buffer, buffer_frames));
    }

    do
    {
        if ((started == false) && (dir == SND_PCM_STREAM_CAPTURE))
        {
            started = true;
            err = snd_pcm_start(pcm);
        }

        if ((started == false) && (dir == SND_PCM_STREAM_PLAYBACK))
        {
            /*side->started=1;*/
            err = PrefillSilence();
        }
        if (err < 0)
            break;/*no recover */

        err = WaitPCM();
        if (err == 0)
            err = RwBuffer(buffer, buffer_frames);

        if (err == -EPIPE || err == -ENODATA)
        {
            tmp_err = Recover(err == -ENODATA);
            if (recover_delay_ms != 0)
            {
                if (recover_delay_ms > 0)
                    usleep(recover_delay_ms * 1000);
                if (recover_delay_ms < 0)
                    break;
            }
            if (tmp_err < 0)
            {
                LOG_ERROR((cply, "RECOVER ERR %d (%s)", tmp_err, snd_strerror(tmp_err)));
                err = tmp_err;
            }
            else
            {
                if (dir == SND_PCM_STREAM_PLAYBACK)
                {
                    /*playback prefill/start on next write call*/
                    err = 0;
                    break;
                }
            }
        }

        if (err == 0)
        {
            /* playback direction should already be started automatically */
            if ((started == 0) && (dir == SND_PCM_STREAM_PLAYBACK))
            {
                started = 1;
                // snd_pcm_start(pcm);		fix SWGIII-8528
            }
        }

        retry_cnt++;
    } while ((err == -EPIPE || err == -ENODATA) && (retry_cnt <= NUM_TRY_RECOVER));

    if (err < 0)
    {
        started = false;
        (void) snd_pcm_drop(pcm);
        broken = true;
        /* Avoid err-log message in upper layer */
        if (aborted)
        {
            err = 0;
        }
    }
    return err;
}

int AlsaAudioCommon::StopStreaming()
{
    int err = 0;

    if (started != true)
    {
        LOG_ERROR((cply, "Can stop streaming as stream is not running"));
        return -EINVAL;
    }

    LOG_INFO((cply, "Stop streaming %s", snd_pcm_name(pcm)));

    startup_xfer = 0;
    if (dir == SND_PCM_STREAM_PLAYBACK)
    {
        err = snd_pcm_nonblock(pcm, 0);
        if (err < 0)
        {
            LOG_ERROR((cply, "Switching %s to blocking mode failed with %d(%s)", snd_pcm_name(pcm),
                    err, snd_strerror(err)));
            broken = true;
            return err;
        }

        err = snd_pcm_drain(pcm);
        if (err < 0)
        {
            LOG_ERROR((cply, "Draining %s failed with %d(%s)", snd_pcm_name(pcm), err,
                    snd_strerror(err)));
            snd_pcm_nonblock(pcm, 1);
            broken = true;
            return err;
        }

        err = snd_pcm_nonblock(pcm, 1);
        if (err < 0)
        {
            LOG_ERROR((cply, "Switching %s to non blocking mode failed with %d(%s)",
                    snd_pcm_name(pcm), err, snd_strerror(err)));
            broken = true;
            return err;
        }
    }
    else
    {
        err = snd_pcm_drop(pcm);
        if (err < 0)
        {
            LOG_ERROR((cply, "Dropping %s failed with %d(%s)", snd_pcm_name(pcm), err,
                    snd_strerror(err)));
            broken = true;
            return err;
        }
    }

    setup = false;
    started = false;
    wait_done = false;

    return err;
}
int AlsaAudioCommon::Delay(snd_pcm_sframes_t *delay)
{
    int err;

    if (pcm == NULL)
    {
        LOG_ERROR((cply, "Failed to get delay as PCM is not opened"));
        return -EINVAL;
    }

    err = snd_pcm_delay(pcm, delay);
    if (err < 0)
    {
        LOG_ERROR((cply, "Delay function of %s failed with %d(%s)", snd_pcm_name(pcm), err,
                snd_strerror(err)));
        return err;
    }
    if (*delay < 0)
    {
        LOG_WARN((cply, "Delay of %s is negative %ld", snd_pcm_name(pcm), *delay));
    }
    if (verbose) {
        LOGD_VERBOSE((cply, "Delay of %s is %ld", snd_pcm_name(pcm), *delay));
    }

    return err;
}

int AlsaAudioCommon::SetThreadPriority(const char* threadName)
{
    int err = 0;

    if(setThreadPrio)
    {
        struct sched_param schedParam;
        int schedPolicy;

        err = pthread_getschedparam(pthread_self(), &schedPolicy, &schedParam);
        if(err == 0)
        {
            schedParam.sched_priority = threadPrio;
            err = pthread_setschedparam(pthread_self(), SCHED_FIFO, &schedParam);
            if(err != 0)
            LOG_ERROR((cply, "%s: set priority failed with error %s", threadName, getErrAsStr(err)));
        }
        else
        {
            LOG_ERROR((cply, "%s: get priority failed with error %s", threadName, getErrAsStr(err)));
        }
    }

    return err;
}

void AlsaAudioCommon::FlushBuffers()
{
    int err = 0;

    if ((pcm != NULL) && (started == true))
    {
        err = snd_pcm_drop(pcm);
        if (err < 0)
        {
            LOG_ERROR((cply, " FlushBuffer:snd_pcm_drop failed with error %s for the device %s", snd_strerror(err), snd_pcm_name(pcm)));
        }
        else
        {
            err = snd_pcm_prepare(pcm);
            if(err < 0)
            {
                LOG_ERROR((cply, " FlushBuffer:snd_pcm_prepare failed with error %s for the device %s", snd_strerror(err), snd_pcm_name(pcm)));
            }
            else
            {
                LOG_INFO((cply, " Alsa: audio buffer flushed"));
            }
        }
    }
}

}
} // namespace adit { namespace carplay
